﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Windows.Forms;
using Tektronix.LogicAnalyzer.Common;
using Tektronix.LogicAnalyzer.TpiNet;

namespace DataWindowPlugInEx
{
    /// <summary>
    /// This is the primary form of the IDataWindowPlugIn.  Please note this form derives
    /// from TlaForm, recommended for any form embedded with the TLA.
    /// </summary>
    public sealed partial class DataWindow : TlaForm
    {
        private const int MinRawColumnWidth = 100;
        private const int MaxRowCount = 100;
        private enum FormatMode { Raw, Logical };

        private DataWindowPlugInEx PlugIn { get; set; }
        private ITlaPlugInSupport Support { get; set; }

        // List of data sources currently tracked by this data window plug-in.
        private readonly List<IStandardDataSource> _dataSources = new List<IStandardDataSource>();

        // The data source currently selected.
        private IStandardDataSource _dataSource;
        public IStandardDataSource DataSource
        {
            get { return _dataSource; }
            private set
            {
                if (_dataSource != value)
                {
                    // If the data source has changed, clear out references to the old IAcquisitionData
                    // and IChannelGrouping objects.
                    if (AcqData != null)
                    {
                        AcqData.ValidityChanged -= AcqDataValidityChanged;
                        AcqData = null;
                    }
                    if (ChannelGrouping != null)
                    {
                        ChannelGrouping.GroupsChanged -= ChannelGroupingGroupsChanged;
                        ChannelGrouping = null;
                    }

                    // Update the data source "selected".
                    _dataSource = value;

                    // If we have one, update the IAcquisitionData and IChannelGrouping objects now.
                    if (_dataSource != null)
                    {
                        AcqData = _dataSource.GetAcquisitionDataObject();
                        if (AcqData != null)
                        {
                            AcqData.ValidityChanged += AcqDataValidityChanged;
                        }
                        ChannelGrouping = _dataSource.GetChannelGroupingObject();
                        if (ChannelGrouping != null)
                        {
                            ChannelGrouping.GroupsChanged += ChannelGroupingGroupsChanged;
                        }
                    }
                    
                    // DataSource has changed, update the controls to reflect it.
                    DefaultDataSourceControls();

                    // Raise the event, in case anyone is listening.
                    if (DataSourceChanged != null)
                    {
                        DataSourceChanged(this, EventArgs.Empty);
                    }
                }
            }
        }

        // The IAcquisitionData and IChannelGrouping objects associated with the current IStandardDataSource.
        // These are set whenever the data source selected changes.
        private IAcquisitionData AcqData { get; set; }
        private IChannelGrouping ChannelGrouping { get; set; }

        // The AcquisitionSummary object for the current IStandardDataSource (must be re-read after a new 
        // acquisition, or when IAcquisition.IsValid toggles).
        private AcquisitionSummary _acqSummary;
        private AcquisitionSummary AcqSummary
        {
            get { return _acqSummary; }
            set
            {
                if (_acqSummary != value)
                {
                    _acqSummary = value;
                    bool hasValidData = _acqSummary != null && _acqSummary.NumberOfSamples > 0;
                    if (_textBoxSample != null)
                    {
                        _textBoxSample.Enabled = hasValidData;
                    }
                    if (_labelSampleOf != null)
                    {
                        _labelSampleOf.Enabled = hasValidData;
                    }
                    if (_dataGrid != null)
                    {
                        _dataGrid.Enabled = hasValidData;
                    }
                }
            }
        }

        // The IRecordAccess object for the current IStandardDataSource (must be re-read after a 
        // new acquisition, or when IAcquisition.IsValid toggles).
        private IRecordAccess _recordAccess;
        private IRecordAccess RecordAccess
        {
            get { return _recordAccess; }
            set
            {
                // If the IRecordAccess object changes, or goes invalid, it is imperative to
                // dispose the old object.
                if (_recordAccess != null)
                {
                    _recordAccess.Dispose();
                }
                _recordAccess = value;
            }
        }

        // The "format" mode that will be used to retrieve data.
        private FormatMode _formatMode = FormatMode.Raw;
        private FormatMode Mode
        {
            get { return _formatMode; }
            set
            {
                if (_formatMode != value)
                {
                    _formatMode = value;
                    UpdateGrid();
                }
            }
        }

        // The starting sample number to retrieve data from.
        private long _startSample = int.MinValue;
        private long StartSample
        {
            get { return _startSample; }
            set
            {
                if (_startSample != value)
                {
                    if (_textBoxSample != null && _labelSampleOf != null)
                    {
                        if (value >= 0)
                        {
                            _startSample = Math.Min(value, AcqSummary.NumberOfSamples - 1);
                            _labelSampleOf.Text = string.Format("of {0} samples", AcqSummary.NumberOfSamples);
                            _labelSampleOf.Enabled = true;
                            _textBoxSample.Text = _startSample.ToString(CultureInfo.InvariantCulture);
                            _textBoxSample.Enabled = true;
                        }
                        else
                        {
                            _startSample = -1;
                            _labelSampleOf.Text = "of ??? samples";
                            _labelSampleOf.Enabled = false;
                            _textBoxSample.Text = "?";
                            _textBoxSample.Enabled = false;
                        }

                        // The data grid will show <x> number of samples, starting from the "starting" sample.
                        UpdateGrid();
                    }
                }
            }
        }

        // Some output to assist with any errors that might occur.
        private string _status = string.Empty;
        private string Status
        {
            set
            {
                if (_status != value)
                {
                    _status = value;
                    if (_labelStatus != null && _toolTip != null)
                    {
                        _toolTip.SetToolTip(_labelStatus, _status);
                        _labelStatus.Text = _status;
                        _labelStatus.Visible = _status != string.Empty;
                    }
                }
            }
        }

        // Event fired whenever the 

        // Constructor.
        public DataWindow(DataWindowPlugInEx plugIn)
        {
            InitializeComponent();

            PlugIn = plugIn;
            if (PlugIn != null)
            {
                Support = PlugIn.Support;
                Text = PlugIn.UserName;
            }
        }

        #region Method(s)

        /// <summary>
        /// Method to initialize the data window.
        /// </summary>
        public void Initialize()
        {
            RegisterEvents();
            InitializeDataSources();
        }

        /// <summary>
        /// Initialize the data source list.
        /// </summary>
        private void InitializeDataSources()
        {
            if (Support != null && _comboBoxDataSources != null)
            {
                // Clear out the existing list(s) of data sources.
                ClearDataSourceList();

                // Re-build the list from the current set of data sources.
                ITlaSystem system = Support.System;
                if (system != null)
                {
                    foreach (object oDataSource in system.DataSources)
                    {
                        var iDataSource = oDataSource as IStandardDataSource;
                        if (iDataSource != null)
                        {
                            AddDataSource(iDataSource);
                        }
                    }
                }

                // The combo box will be enabled if there are any sources to select.
                _comboBoxDataSources.Enabled = _comboBoxDataSources.Items.Count > 0;

                // Next, make sure the "selected" data source (if we had one), is still there.
                // If not, remove any reference to it now.
                if (DataSource != null && !_dataSources.Contains(DataSource))
                {
                    DataSource = null;
                }

                // Lastly, update the combo selection, if we have a match.
                if (DataSource != null && _comboBoxDataSources.Items.Contains(DataSource.UserName))
                {
                    _comboBoxDataSources.SelectedItem = DataSource.UserName;
                }
            }
        }

        /// <summary>
        /// Register events (called during Initialize()).
        /// </summary>
        private void RegisterEvents()
        {
            if (PlugIn != null)
            {
                PlugIn.UserNameChanged += PlugInUserNameChanged;
            }
            if (_comboBoxDataSources != null)
            {
                _comboBoxDataSources.EnabledChanged += ComboBoxDataSourcesEnabledChanged;
                _comboBoxDataSources.SelectedValueChanged += ComboBoxDataSourcesSelectedValueChanged;
            }
            if (_radioRaw != null)
            {
                _radioRaw.CheckedChanged += RadioModeCheckedChanged;
            }
            if (_radioLogical != null)
            {
                _radioLogical.CheckedChanged += RadioModeCheckedChanged;
            }
            if (_textBoxSample != null)
            {
                _textBoxSample.LostFocus += TextBoxSampleLostFocus;
                _textBoxSample.KeyDown += TextBoxSampleKeyDown;
            }
            if (_dataGrid != null)
            {
                _dataGrid.SizeChanged += DataGridSizeChanged;
            }
        }

        /// <summary>
        /// Unregister events (called during Dispose()).
        /// </summary>
        private void UnregisterEvents()
        {
            if (PlugIn != null)
            {
                PlugIn.UserNameChanged -= PlugInUserNameChanged;
            }
            if (_comboBoxDataSources != null)
            {
                _comboBoxDataSources.EnabledChanged -= ComboBoxDataSourcesEnabledChanged;
                _comboBoxDataSources.SelectedValueChanged -= ComboBoxDataSourcesSelectedValueChanged;
            }
            if (_radioRaw != null)
            {
                _radioRaw.CheckedChanged -= RadioModeCheckedChanged;
            }
            if (_radioLogical != null)
            {
                _radioLogical.CheckedChanged -= RadioModeCheckedChanged;
            }
            if (_textBoxSample != null)
            {
                _textBoxSample.TextChanged -= TextBoxSampleLostFocus;
                _textBoxSample.KeyDown -= TextBoxSampleKeyDown;
            }
            if (_dataGrid != null)
            {
                _dataGrid.SizeChanged -= DataGridSizeChanged;
            }
        }

        /// <summary>
        /// Clears both the data source list and drop-down list associated with them.
        /// </summary>
        private void ClearDataSourceList()
        {
            if (_comboBoxDataSources != null)
            {
                _comboBoxDataSources.Items.Clear();
            }
            if (_dataSources != null)
            {
                foreach (IStandardDataSource dataSource in _dataSources)
                {
                    dataSource.UserNameChanged -= DataSourceUserNameChanged;
                }
                _dataSources.Clear();
            }
        }

        /// <summary>
        /// Adds the given data source to the combo list, and starts listening for name changes.
        /// </summary>
        /// <param name="dataSource"></param>
        private void AddDataSource(IStandardDataSource dataSource)
        {
            if (dataSource != null)
            {
                if (_dataSources != null && _comboBoxDataSources != null)
                {
                    dataSource.UserNameChanged += DataSourceUserNameChanged;
                    _comboBoxDataSources.Items.Add(dataSource.UserName);
                    _dataSources.Add(dataSource);
                }
            }
        }

        /// <summary>
        /// Finds the data source, given the user name displayed in the combo drop-down.
        /// </summary>
        /// <param name="userName"></param>
        /// <returns></returns>
        private IStandardDataSource FindDataSource(string userName)
        {
            if (_dataSources != null)
            {
                foreach (IStandardDataSource dataSource in _dataSources)
                {
                    // Return the first match.  These should all be unique, as dictated by the TLA.
                    if (dataSource.UserName == userName)
                    {
                        return dataSource;
                    }
                }
            }

            // Didn't find a match.
            return null;
        }

        /// <summary>
        /// This method should be called during initialization or whenever the IAcquisitionData 
        /// throws the ValidityChanged event.
        /// </summary>
        private void DefaultDataSourceControls()
        {
            if (_comboBoxDataSources != null)
            {
                // The remaining controls will be enabled if there is a data source selected
                // and valid data contained within that data source.
                if (DataSource != null)
                {
                    // Get the acquisition data object for the selected data source, and verify the data
                    // is good (you cannot and should not get the AcquisitionSummary object is IsValid
                    // is false for IAqcuisitionData.
                    AcqSummary = AcqData != null && AcqData.IsValid ? AcqData.TimebaseSummary(DataSetValue.Main) : null;

                    // The IRecordAccess object also needs to be re-created, after IAcquisitionData.ValidityChanged.
                    RecordAccess = AcqData != null && AcqData.IsValid ? AcqData.CreateRecordAccessObject() : null;
                }

                // The format mode radio buttons should now be updated.
                DefaultFormatMode();

                // The "SampleOf" label can be used to show the total number of samples for the 
                // current acquisition, if the data source has valid data (it is hidden otherwise).
                DefaultStartSample();

                // Finally, update the grid.
                UpdateGrid();
            }
        }

        /// <summary>
        /// Updates the radio buttons representing the format mode to extract data.
        /// </summary>
        private void DefaultFormatMode()
        {
            if (_radioRaw != null && _radioLogical != null)
            {
                if (ChannelGrouping != null)
                {
                    _radioRaw.Enabled = true;
                    _radioLogical.Enabled = ChannelGrouping.Groups.Count > 0;
                    if (ChannelGrouping.Groups.Count == 0)
                    {
                        _radioLogical.Checked = false;
                        _radioRaw.Checked = true;
                    }
                }
                else
                {
                    _radioRaw.Enabled = false;
                    _radioLogical.Enabled = false;
                }
            }
        }

        /// <summary>
        /// Updates the "sample" controls.
        /// </summary>
        private void DefaultStartSample()
        {
            if (AcqSummary != null && AcqSummary.NumberOfSamples > 0)
            {
                StartSample = AcqSummary.HasTriggerSample ? AcqSummary.TriggerSample : 0;
            }
            else
            {
                StartSample = -1;
            }
        }

        /// <summary>
        /// This method takes the textbox value of the "start sample", and converts it into
        /// a valid starting sample (long).
        /// </summary>
        private void UpdateStartSample()
        {
            if (_textBoxSample != null)
            {
                try
                {
                    StartSample = Convert.ToInt64(_textBoxSample.Text);
                }
                catch (FormatException)
                {
                    _textBoxSample.Text = StartSample >= 0 ? StartSample.ToString(CultureInfo.InvariantCulture) : "?";
                }
                catch (OverflowException)
                {
                    _textBoxSample.Text = StartSample >= 0 ? StartSample.ToString(CultureInfo.InvariantCulture) : "?";
                }
            }
        }

        /// <summary>
        /// Updates the data source "grid", showing raw data, timestamp, etc.
        /// </summary>
        private void UpdateGrid()
        {
            if (_dataGrid != null && _radioRaw != null && _radioLogical != null)
            {
                // Clear out any existing status message.
                Status = string.Empty;

                // Clear out any existing row data.
                _dataGrid.RowCount = 0;

                // Check for valid data.
                if (AcqData != null && AcqSummary != null && AcqSummary.NumberOfSamples > 0 && RecordAccess != null)
                {
                    // Prepare the IRecordAccess object to request data.
                    if (PrepareRecordAccess())
                    {
                        // How many rows can we show (make sure we don't run off the end of the acquisition).
                        long maxRowLong = Math.Min(AcqSummary.NumberOfSamples - StartSample, int.MaxValue);
                        _dataGrid.RowCount = Math.Min(MaxRowCount, (int) maxRowLong);

                        // Request the number of samples we're going to create rows for (1:1 match, in this case).
                        var records = (byte[,]) RecordAccess.GetByteRecords(StartSample, _dataGrid.RowCount);

                        // Update the grid columns - sample #, data value, timestamp value
                        for (int i = 0; i < _dataGrid.RowCount; i++)
                        {
                            var sample = i + StartSample;
                            _dataGrid.Rows[i].Cells[0].Value = sample;
                            _dataGrid.Rows[i].Cells[1].Value = GetSampleValue(records, i);
                            _dataGrid.Rows[i].Cells[2].Value = AcqData.GetTimestamp(sample, DataSetValue.Main);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// This method prepares the IRecordAccess object to request data.
        /// </summary>
        /// <returns>TRUE if okay to retrieve data, FALSE otherwise.</returns>
        private bool PrepareRecordAccess()
        {
            if (RecordAccess != null)
            {
                // Tell the record access object which type of data to return.
                RecordAccess.DataSet = DataSetValue.Main;

                // Two methods of access: 
                // 1) "Raw" mode, which will return all signal data in the raw acquisition format.
                if (Mode == FormatMode.Raw)
                {
                    RecordAccess.FormatRaw();
                    return true;
                }

                // 2) "Logical" mode, which will group all data together by the groups specified.
                if (Mode == FormatMode.Logical) 
                {
                    // If no channel grouping object, we cannot use FormatLogical().
                    if (ChannelGrouping == null || ChannelGrouping.Groups.Count == 0)
                    {
                        Status = "No IChannelGrouping object is found. Please ensure the selected data source has one or more valid groups defined.";
                        return false;
                    }

                    // Build of the list of group names you're interested in.
                    var groups = new List<string>();
                    foreach (IChannelGroup group in ChannelGrouping.Groups)
                    {
                        groups.Add(@group.Name);
                    }
                    try
                    {
                        // This behavior just grabs all currently defined groups.  In reality, one could use a 
                        // list of any valid groups (assuming they are defined and contain at least one signal).
                        // e.g. ==> IRecordAccess.FormatLogical(new string[] { "group_A2", "group_A3" });
                        RecordAccess.FormatLogical(groups.ToArray());
                        return true;
                    }
                    catch (ArgumentException)
                    {
                        // If any group is empty, a ArgumentException will be thrown.
                        // Display some info to the user, and bail out.
                        Status = "IRecordAccess threw ArgumentException. Please select \"Raw\" format mode, or ensure that all groups defined contain at least one signal.";
                        return false;
                    }
                }
            }

            // Something didn't work out.  Return FALSE to indicate IRecordAccess not ready
            // to request data.
            return false;
        }

        /// <summary>
        /// Helper method to convert the raw/logical data value, into a string for display in the grid.
        /// </summary>
        /// <param name="records"></param>
        /// <param name="sampleIndex"></param>
        /// <returns></returns>
        private string GetSampleValue(byte[,] records, long sampleIndex)
        {
            string sampleValue = string.Empty;

            // The first index is the sample number where the data is located.  Make sure the
            // sample requested is valid for the give byte[,] object.
            if (records != null && sampleIndex < records.GetLength(0))
            {
                // The second index is the number of bytes for that sample.
                int sampleLength = records.GetLength(1);
                for (int byteIndex = 0; byteIndex < sampleLength; byteIndex++)
                {
                    // Convert byte value to hexidecimal string value for display.
                    sampleValue += records[sampleIndex, byteIndex].ToString("X").PadLeft(2, '0');
                }
            }

            return sampleValue;
        }

        #endregion

        #region Event(s)

        public event EventHandler DataSourceChanged;

        #endregion

        #region Event Handler(s)

        /// <summary>
        /// The user name of the data window has changed.  Update the window "Text" or title bar.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PlugInUserNameChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (PlugIn != null)
            {
                Text = PlugIn.UserName;
            }
        }

        /// <summary>
        /// If the combo box for data sources is disabled, all other controls should also be disabled.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ComboBoxDataSourcesEnabledChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            DefaultDataSourceControls();
        }

        /// <summary>
        /// The user has changed his/her selected data source.  Update that now.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ComboBoxDataSourcesSelectedValueChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (_comboBoxDataSources != null && _comboBoxDataSources.SelectedItem != null)
            {
                DataSource = FindDataSource(_comboBoxDataSources.SelectedItem.ToString());
            }
            else
            {
                DataSource = null;
            }
        }

        /// <summary>
        /// Anytime one of the "mode" checkboxes change state, update the mode of operation.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void RadioModeCheckedChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (_radioLogical != null)
            {
                Mode = _radioLogical.Checked ? FormatMode.Logical : FormatMode.Raw;
            }
        }

        /// <summary>
        /// After losing focus, update the starting sample.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TextBoxSampleLostFocus(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            UpdateStartSample();
        }

        /// <summary>
        /// Look for an "Enter" keypress.  If found, update the starting sample.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TextBoxSampleKeyDown(object sender, KeyEventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (e.KeyCode == Keys.Enter)
            {
                UpdateStartSample();
            }
        }

        /// <summary>
        /// This method is mainly for aesthetics, to keep
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DataGridSizeChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (_dataGrid != null && ColumnData != null)
            {
                int width = _dataGrid.Width - _dataGrid.RowHeadersWidth - ColumnSample.Width - ColumnTimestamp.Width - 19;
                ColumnData.Width = Math.Max(width, MinRawColumnWidth);
            }
        }

        /// <summary>
        /// A data source's user name has changed.  Update the list now.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="eventArgs"></param>
        private void DataSourceUserNameChanged(object sender, EventArgs eventArgs)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            InitializeDataSources();
        }

        /// <summary>
        /// The selected data source's acquisition data has toggled its validity.  Update the
        /// controls are currently showing/not-showing data.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void AcqDataValidityChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            DefaultDataSourceControls();
        }

        /// <summary>
        /// The selected data source has had a group added or removed from it.  We track this because
        /// we need to make sure there are groups, otherwise IRecordAccess.FormatLogical() would 
        /// not be possible.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ChannelGroupingGroupsChanged(object sender, ObjectEventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            DefaultFormatMode();
        }

        #endregion

        #region IDisposable Member(s)

        /// <summary>
        /// Disposal flag.
        /// </summary>
        private bool _disposed;

        /// <summary> 
        /// Clean up any resources being used.
        /// </summary>
        /// <param name="disposing">true if managed resources should be disposed; otherwise, false.</param>
        protected override void Dispose(bool disposing)
        {
            if (!_disposed)
            {
                try
                {
                    if (disposing)
                    {
                        // Unregister from all events.
                        UnregisterEvents();

                        // Null out the selected data source.  For this plug-in, this call will
                        // also dispose the IRecordAccess object, if one has been created (which
                        // is important to do).
                        DataSource = null;

                        // Now, dispose/clear any other objects that implement IDisposable.
                        if (components != null)
                        {
                            components.Dispose();
                            components = null;
                        }
                    }
                    base.Dispose(disposing);
                }
                catch
                {
                    _disposed = false;
                    throw;
                }
                _disposed = true;
            }
        }

        #endregion
    }
}
